/*
 * Decompiled with CFR 0.152.
 */
package jenkins.security.apitoken;

import edu.umd.cs.findbugs.annotations.CheckForNull;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import hudson.Util;
import hudson.util.Secret;
import java.io.Serializable;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import jenkins.security.Messages;
import jenkins.security.apitoken.TokenUuidAndPlainValue;
import net.jcip.annotations.Immutable;
import net.sf.json.JSONObject;
import org.kohsuke.accmod.Restricted;
import org.kohsuke.accmod.restrictions.NoExternalUse;

@Restricted(value={NoExternalUse.class})
public class ApiTokenStore {
    private static final Logger LOGGER = Logger.getLogger(ApiTokenStore.class.getName());
    private static final SecureRandom RANDOM = new SecureRandom();
    private static final Comparator<HashedToken> SORT_BY_LOWERCASED_NAME = Comparator.comparing(hashedToken -> hashedToken.getName().toLowerCase(Locale.ENGLISH));
    private static final int TOKEN_LENGTH_V2 = 34;
    private static final String LEGACY_VERSION = "10";
    private static final String HASH_VERSION = "11";
    private static final String HASH_ALGORITHM = "SHA-256";
    private List<HashedToken> tokenList;
    private static final int VERSION_LENGTH = 2;
    private static final int HEX_CHAR_LENGTH = 32;
    private static final Pattern CHECK_32_HEX_CHAR = Pattern.compile("[a-f0-9]{32}");

    public ApiTokenStore() {
        this.init();
    }

    private Object readResolve() {
        this.init();
        return this;
    }

    private void init() {
        if (this.tokenList == null) {
            this.tokenList = new ArrayList<HashedToken>();
        }
    }

    @NonNull
    public synchronized Collection<HashedToken> getTokenListSortedByName() {
        return this.tokenList.stream().sorted(SORT_BY_LOWERCASED_NAME).collect(Collectors.toList());
    }

    private void addToken(HashedToken token) {
        this.tokenList.add(token);
    }

    public synchronized void reconfigure(@NonNull Map<String, JSONObject> tokenStoreDataMap) {
        this.tokenList.forEach(hashedToken -> {
            JSONObject receivedTokenData = (JSONObject)tokenStoreDataMap.get(hashedToken.uuid);
            if (receivedTokenData == null) {
                LOGGER.log(Level.INFO, "No token received for {0}", hashedToken.uuid);
                return;
            }
            String name = receivedTokenData.getString("tokenName");
            if (name == null || name.isBlank()) {
                LOGGER.log(Level.INFO, "Empty name received for {0}, we do not care about it", hashedToken.uuid);
                return;
            }
            hashedToken.setName(name);
        });
    }

    public synchronized void regenerateTokenFromLegacy(@NonNull Secret newLegacyApiToken) {
        this.deleteAllLegacyAndGenerateNewOne(newLegacyApiToken, false);
    }

    public synchronized void regenerateTokenFromLegacyIfRequired(@NonNull Secret newLegacyApiToken) {
        if (this.tokenList.stream().noneMatch(HashedToken::isLegacy)) {
            this.deleteAllLegacyAndGenerateNewOne(newLegacyApiToken, true);
        }
    }

    private void deleteAllLegacyAndGenerateNewOne(@NonNull Secret newLegacyApiToken, boolean migrationFromExistingLegacy) {
        this.deleteAllLegacyTokens();
        this.addLegacyToken(newLegacyApiToken, migrationFromExistingLegacy);
    }

    private void deleteAllLegacyTokens() {
        this.tokenList.removeIf(HashedToken::isLegacy);
    }

    private void addLegacyToken(@NonNull Secret legacyToken, boolean migrationFromExistingLegacy) {
        String tokenUserUseNormally = Util.getDigestOf(legacyToken.getPlainText());
        String secretValueHashed = this.plainSecretToHashInHex(tokenUserUseNormally);
        HashValue hashValue = new HashValue(LEGACY_VERSION, secretValueHashed);
        HashedToken token = HashedToken.buildNewFromLegacy(hashValue, migrationFromExistingLegacy);
        this.addToken(token);
    }

    @Nullable
    public synchronized HashedToken getLegacyToken() {
        return this.tokenList.stream().filter(HashedToken::isLegacy).findFirst().orElse(null);
    }

    @NonNull
    public synchronized TokenUuidAndPlainValue generateNewToken(@NonNull String name) {
        byte[] random = new byte[16];
        RANDOM.nextBytes(random);
        String secretValue = Util.toHexString(random);
        String tokenTheUserWillUse = HASH_VERSION + secretValue;
        assert (tokenTheUserWillUse.length() == 34);
        HashedToken token = this.prepareAndStoreToken(name, secretValue);
        return new TokenUuidAndPlainValue(token.uuid, tokenTheUserWillUse);
    }

    @SuppressFBWarnings(value={"UNSAFE_HASH_EQUALS"}, justification="Comparison only validates version of the specified token")
    @NonNull
    public synchronized String addFixedNewToken(@NonNull String name, @NonNull String tokenPlainValue) {
        if (tokenPlainValue.length() != 34) {
            LOGGER.log(Level.INFO, "addFixedNewToken, length received: {0}" + tokenPlainValue.length());
            throw new IllegalArgumentException("The token must consist of 2 characters for the version and 32 hex-characters for the secret");
        }
        String hashVersion = tokenPlainValue.substring(0, 2);
        if (!HASH_VERSION.equals(hashVersion)) {
            throw new IllegalArgumentException("The given version is not recognized: " + hashVersion);
        }
        String tokenPlainHexValue = tokenPlainValue.substring(2);
        if (!CHECK_32_HEX_CHAR.matcher(tokenPlainHexValue = tokenPlainHexValue.toLowerCase()).matches()) {
            throw new IllegalArgumentException("The secret part of the token must consist of 32 hex-characters");
        }
        HashedToken token = this.prepareAndStoreToken(name, tokenPlainHexValue);
        return token.uuid;
    }

    @NonNull
    private HashedToken prepareAndStoreToken(@NonNull String name, @NonNull String tokenPlainValue) {
        String secretValueHashed = this.plainSecretToHashInHex(tokenPlainValue);
        HashValue hashValue = new HashValue(HASH_VERSION, secretValueHashed);
        HashedToken token = HashedToken.buildNew(name, hashValue);
        this.addToken(token);
        return token;
    }

    @NonNull
    private String plainSecretToHashInHex(@NonNull String secretValueInPlainText) {
        byte[] hashBytes = this.plainSecretToHashBytes(secretValueInPlainText);
        return Util.toHexString(hashBytes);
    }

    @NonNull
    private byte[] plainSecretToHashBytes(@NonNull String secretValueInPlainText) {
        return this.hashedBytes(secretValueInPlainText.getBytes(StandardCharsets.US_ASCII));
    }

    @NonNull
    private byte[] hashedBytes(byte[] tokenBytes) {
        MessageDigest digest;
        try {
            digest = MessageDigest.getInstance(HASH_ALGORITHM);
        }
        catch (NoSuchAlgorithmException e) {
            throw new AssertionError("There is no SHA-256 available in this system", e);
        }
        return digest.digest(tokenBytes);
    }

    @CheckForNull
    public synchronized HashedToken findMatchingToken(@NonNull String token) {
        String plainToken = this.isLegacyToken(token) ? token : this.getHashOfToken(token);
        return this.searchMatch(plainToken);
    }

    private boolean isLegacyToken(@NonNull String token) {
        return token.length() != 34;
    }

    @NonNull
    private String getHashOfToken(@NonNull String token) {
        return token.substring(2);
    }

    @CheckForNull
    private HashedToken searchMatch(@NonNull String plainSecret) {
        byte[] hashedBytes = this.plainSecretToHashBytes(plainSecret);
        for (HashedToken token : this.tokenList) {
            if (!token.match(hashedBytes)) continue;
            return token;
        }
        return null;
    }

    @SuppressFBWarnings(value={"UNSAFE_HASH_EQUALS"}, justification="Only used during revocation.")
    @CheckForNull
    public synchronized HashedToken revokeToken(@NonNull String tokenUuid) {
        Iterator<HashedToken> iterator = this.tokenList.iterator();
        while (iterator.hasNext()) {
            HashedToken token = iterator.next();
            if (!token.uuid.equals(tokenUuid)) continue;
            iterator.remove();
            return token;
        }
        return null;
    }

    public synchronized void revokeAllTokens() {
        this.tokenList.clear();
    }

    public synchronized void revokeAllTokensExcept(@NonNull String tokenUuid) {
        this.tokenList.removeIf(token -> !token.uuid.equals(tokenUuid));
    }

    public synchronized boolean renameToken(@NonNull String tokenUuid, @NonNull String newName) {
        for (HashedToken token : this.tokenList) {
            if (!token.uuid.equals(tokenUuid)) continue;
            token.rename(newName);
            return true;
        }
        LOGGER.log(Level.FINER, "The target token for rename does not exist, for uuid = {0}, with desired name = {1}", new Object[]{tokenUuid, newName});
        return false;
    }

    @Immutable
    private static class HashValue
    implements Serializable {
        private static final long serialVersionUID = 1L;
        private final String version;
        private final String hash;

        private HashValue(String version, String hash) {
            this.version = version;
            this.hash = hash;
        }
    }

    public static class HashedToken
    implements Serializable {
        private static final long serialVersionUID = 1L;
        private String uuid;
        private String name;
        private Date creationDate;
        private HashValue value;

        private HashedToken() {
            this.init();
        }

        private Object readResolve() {
            this.init();
            return this;
        }

        private void init() {
            if (this.uuid == null) {
                this.uuid = UUID.randomUUID().toString();
            }
        }

        @NonNull
        public static HashedToken buildNew(@NonNull String name, @NonNull HashValue value) {
            HashedToken result = new HashedToken();
            result.name = name;
            result.creationDate = new Date();
            result.value = value;
            return result;
        }

        @NonNull
        public static HashedToken buildNewFromLegacy(@NonNull HashValue value, boolean migrationFromExistingLegacy) {
            HashedToken result = new HashedToken();
            result.name = Messages.ApiTokenProperty_LegacyTokenName();
            result.creationDate = migrationFromExistingLegacy ? null : new Date();
            result.value = value;
            return result;
        }

        public void rename(String newName) {
            this.name = newName;
        }

        public boolean match(byte[] hashedBytes) {
            byte[] hashFromHex;
            try {
                hashFromHex = Util.fromHexString(this.value.hash);
            }
            catch (NumberFormatException e) {
                LOGGER.log(Level.INFO, "The API token with name=[{0}] is not in hex-format and so cannot be used", this.name);
                return false;
            }
            return MessageDigest.isEqual(hashFromHex, hashedBytes);
        }

        public String getName() {
            return this.name;
        }

        public Date getCreationDate() {
            return this.creationDate;
        }

        public long getNumDaysCreation() {
            return this.creationDate == null ? 0L : Util.daysElapsedSince(this.creationDate);
        }

        public String getUuid() {
            return this.uuid;
        }

        public boolean isLegacy() {
            return this.value.version.equals(ApiTokenStore.LEGACY_VERSION);
        }

        public void setName(String name) {
            this.name = name;
        }
    }
}

